/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: RS232Layer.h 5609 2013-08-06 14:13:43Z sebastiand $
   Last Modification:
    $Author: sebastiand $
    $Date: 2013-08-06 16:13:43 +0200 (Di, 06 Aug 2013) $
    $Revision: 5609 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
     Physical layer for UART interfaces

   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         25.11.10    MT       Added CfgMgr32 Function to get status of detected
                                    devices
     2         27.11.09    SS       Review
     1         03.03.09    MT       initial version
 
**************************************************************************************/

/*****************************************************************************/
/*! \file RS232Layer.h
*   Physical layer for UART interfaces                                       */
/*****************************************************************************/

#pragma once

#include <string>
#pragma warning(push)
#pragma warning(disable: 4702)
#include <map>
#include <vector>
#pragma warning(pop)

#include "Dbt.h"
#include "RS232Config.h"
#include "connectorAPI.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_RS232 netX RS232 Connector                    */
/*! \{                                                                       */
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
/// Definitions
///////////////////////////////////////////////////////////////////////////////////////////
#define RS232_CONNECTOR_IDENTIFIER  "COM"               /*!< Identifier of the RS232 connector  */
#define RS232_CONNECTOR_DESCRIPTION "USB/RS232 Connection" /*!< Description of the RS232 connector */

class CRS232Interface;

/*! Structure holding interface configuration */
typedef struct INTERFACE_CONTROL_Ttag
{
  unsigned long         ulState;            /*!< State of interface                   */
  DWORD                 dwBaudRate;         /*!< Baudrate used by the interface       */
  BYTE                  bByteSize;          /*!< Bytesize used by the interface       */
  BYTE                  bParity;            /*!< Parity used by the interface         */
  BYTE                  bStopBits;          /*!< Stopbits used by the interface       */
  DWORD                 dwResetTimeout;     /*!< Reset Timeout for the interface      */
  DWORD                 dwSendTimeout;      /*!< Send Timeout for the interface       */
  DWORD                 dwKeepAliveTimeout; /*!< Keep Alive Timeout for the interface */

} INTERFACE_CONTROL_T, *PINTERFACE_CONTROL_T;

/*! Map holding the interfaces created by the connector */
typedef std::map<std::string, CRS232Interface*> INTERFACE_MAP;

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CRS232Layer
/// Physical Layer for UART interfaces
///////////////////////////////////////////////////////////////////////////////////////////
class CRS232Layer
{
public:
  CRS232Layer(void);
  ~CRS232Layer(void);

  CRS232Interface*        GetInterface(std::string szDeviceName);

  long                    Init              ( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser);
  void                    Deinit            ( void);
  void                    Reinit            ( void);
  long                    ValidateInterface ( PCONNECTOR_INTERFACE pvInterface);
  
  CRITICAL_SECTION        m_tcsInterfaceLock;             /*!< Critical Section for the interface map         */      

protected:
  HANDLE                  m_hDeviceEnumThread;            /*!< Thread for device enumeration                  */
  HANDLE                  m_hDeviceEnumThreadReady;       /*!< Device enumeration thread signals ready        */
  HWND                    m_hwndDeviceEnum;               /*!< Window receiving device manager notifications  */
  std::vector<HDEVNOTIFY> m_cvDeviceNotifications;        /*!< All registered device notifications            */
  
  INTERFACE_MAP           m_cmInterfaceControlMap;        /*!< All available interfaces                       */
  HANDLE                  m_hControlInterfaceThread;      /*!< Thread for interface control                   */
  HANDLE                  m_hControlInterfaceStop;        /*!< Control interface thread signal stop           */
  HANDLE                  m_hControlInterfaceReady;       /*!< Control interface thread signal ready          */
  HANDLE                  m_hControlInterfaceNotifyEvent; /*!< Signals Notifcation Event of the interface     */

                              

  PFN_NETXCON_DEVICE_NOTIFY_CALLBACK m_pfnNotifyCallback; /*!< Callback for the notify calls                  */
  void*                              m_pvNotifyUser;      /*!< User pointer for the notify calls              */

  static DWORD WINAPI     DeviceEnumThread(void* pvParam);/*!< Thread for enumeration of devices              */
  static DWORD WINAPI     ControlInterfaceThread(void* pvParam);  /*!< Thread for Interface control           */
  

  static LRESULT CALLBACK HandleWindowProc   ( HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam); 
  bool                    ExtractDeviceName(PDEV_BROADCAST_DEVICEINTERFACE pDevInf, WPARAM wParam, std::string& szDeviceName);
  void                    EnumerateDevices(void);         /*!< Enumerate available devices                    */
  long                    EnableDeviceNotification();     /*!< Enable device notification                     */
  void                    DisableDeviceNotification();    /*!< Disable devices notification                   */

  void                    CreateInterface(std::string szDeviceName);
private:
  
};

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CRS232Interface
/// Physical interface driver for a single COM port
///////////////////////////////////////////////////////////////////////////////////////////
class CRS232Interface
{
  friend class CRS232Layer;
private:
  CRS232Interface();          

public:
  CRS232Interface  ( const char* szIntfName, INTERFACE_CONTROL_T* ptInterfaceControlData);
  ~CRS232Interface ();

  long                  Start( PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveCallback, 
                               void*                               pvUser);

  void                  Stop ( void);

  long                  Send ( unsigned char*       pabData,
                               unsigned long        ulDataLen);

  long                  GetInterfaceName      ( unsigned long ulSize, char* szInterface);

  //////////////////////////////////////////////////
  /// Get state of the interface 
  ///   \return State of interface (NETX_INTERFACE_STATE_E)
  //////////////////////////////////////////////////
  unsigned long         GetCtrlState          ( void)                  { return m_tInterfaceControl.ulState;} 

  //////////////////////////////////////////////////
  /// Get send timeout of the interface
  ///   \return Send timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetSendTimeout        ( void)                  { return m_tInterfaceControl.dwSendTimeout;} 

  //////////////////////////////////////////////////
  /// Get reset timeout of the interface
  ///   \return Reset timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetResetTimeout       ( void)                  { return m_tInterfaceControl.dwResetTimeout;} 

  //////////////////////////////////////////////////
  /// Get keep alive timeout of the interface
  ///   \return Keep alive timeout in ms
  //////////////////////////////////////////////////
  unsigned long         GetKeepAliveTimeout   ( void)                  { return m_tInterfaceControl.dwKeepAliveTimeout;} 

  //////////////////////////////////////////////////
  /// Get baud rate of the interface
  ///   \return e.g. CBR_115200
  //////////////////////////////////////////////////
  DWORD                 GetBaudRate(void)   { return m_tInterfaceControl.dwBaudRate; }

  //////////////////////////////////////////////////
  /// Get byte size of the interface
  ///   \return The number of bits in the bytes transmitted and received
  //////////////////////////////////////////////////
  BYTE                  GetByteSize(void)   { return m_tInterfaceControl.bByteSize;  }

  //////////////////////////////////////////////////
  /// Get parity of the interface
  ///   \return NOPARITY, ODDPARITY, EVENPARITY, MARKPARITY or SPACEPARITY
  //////////////////////////////////////////////////
  BYTE                  GetParity(void)     { return m_tInterfaceControl.bParity;    }

  //////////////////////////////////////////////////
  /// Get stop bits of the interface
  ///   \return ONESTOPBIT, ONE5STOPBITS or TWOSTOPBITS
  //////////////////////////////////////////////////
  BYTE                  GetStopBits(void)   { return m_tInterfaceControl.bStopBits;  }

  //////////////////////////////////////////////////
  /// Set state of interface 
  ///   \param ulState New interface state (NETX_INTERFACE_STATE_E)
  //////////////////////////////////////////////////
  void                  SetCtrlState ( unsigned long ulState) { m_tInterfaceControl.ulState = ulState;} 

  //////////////////////////////////////////////////
  /// Set send timeout of the interface
  ///   \param dwSendTimeout Send timeout in ms
  //////////////////////////////////////////////////
  void                  SetSendTimeout        ( DWORD dwSendTimeout)      { m_tInterfaceControl.dwSendTimeout = dwSendTimeout;} 

  //////////////////////////////////////////////////
  /// Set reset timeout of the interface
  ///   \param dwResetTimeout Reset timeout in ms
  //////////////////////////////////////////////////
  void                  SetResetTimeout       ( DWORD dwResetTimeout)     { m_tInterfaceControl.dwResetTimeout = dwResetTimeout;} 

  //////////////////////////////////////////////////
  /// Set keep alive timeout of the interface
  ///   \param dwKeepAliveTimeout Keep alive timeout in ms
  //////////////////////////////////////////////////
  void                  SetKeepAliveTimeout   ( DWORD dwKeepAliveTimeout) { m_tInterfaceControl.dwKeepAliveTimeout = dwKeepAliveTimeout;} 

  //////////////////////////////////////////////////
  /// Set baud rate of the interface 
  ///   \param dwBaudrate e.g. CBR_115200
  //////////////////////////////////////////////////
  void                  SetBaudRate(DWORD dwBaudrate) { m_tInterfaceControl.dwBaudRate = dwBaudrate; }

  //////////////////////////////////////////////////
  /// Set byte size of the interface 
  ///   \param bByteSize The number of bits in the bytes transmitted and received
  //////////////////////////////////////////////////
  void                  SetByteSize(BYTE bByteSize)   { m_tInterfaceControl.bByteSize  = bByteSize;  }

  //////////////////////////////////////////////////
  /// Set parity of the interface 
  ///   \param bParity NOPARITY, ODDPARITY, EVENPARITY, MARKPARITY or SPACEPARITY
  //////////////////////////////////////////////////
  void                  SetParity(BYTE bParity)       { m_tInterfaceControl.bParity    = bParity;    }

  //////////////////////////////////////////////////
  /// Set stop bits of the interface 
  ///   \param bStopBits ONESTOPBIT, ONE5STOPBITS or TWOSTOPBITS
  //////////////////////////////////////////////////
  void                  SetStopBits(BYTE bStopBits)   { m_tInterfaceControl.bStopBits  = bStopBits;  }

protected:
                                                          
  std::string           m_szDeviceName;                     /*!< Name of the Interface                          */
  HANDLE                m_hComPort;                         /*!< Handle of COM-Port                             */
  HANDLE                m_hComThread;                       /*!< Handle of COM Thread                           */
  HANDLE                m_hComThreadReady;                  /*!< Signal handle COM thread is ready              */
  HANDLE                m_hComThreadStop;                   /*!< Stop-Event of the COM Thread                   */
  
  BYTE                  m_abRecvBuffer[2048];
  INTERFACE_CONTROL_T   m_tInterfaceControl;                /*!< Interface control data structure               */

  BOOL                  m_fNotify;                          /*!< TRUE notify interface                          */

  PFN_NETXCON_DEVICE_RECEIVE_CALLBACK m_pfnReceiveCallback; /*!< Receive-Callback ( Function-Callback))        */
  void*                               m_pvReceiveUserData;  /*!< Received data                                 */

  CRITICAL_SECTION      m_tcsInterfaceLock;

  static DWORD WINAPI   CommThread     ( void* pvParam);    /*!< COM-Thread                                    */
  DWORD                 CommThreadFunc ( void);             /*!< Start COM-Thread                              */


  long                  OpenPort       ( void);             /*!< Open COM-Port, belong given name of interface */

  long                  SetupConnection( DWORD dwBaudRate,
                                         BYTE  bByteSize,
                                         BYTE  bParity,
                                         BYTE  bStopBits);  /*!< Configurate Serial-Connection                 */

  BOOL UpdateInterfaceState(void)         { return m_fNotify;    } /*!< Checks if it is alloed to update the interface state */
  void UpdateInterfaceState(BOOL fNotify) { m_fNotify = fNotify; } /*!< En-/disables update of the interface state change    */

};

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
